<?php
/**
 * Envo Royal Child — clean enqueue order + cache-busting
 * Put this entire file as your child theme's functions.php
 */

/* ------------------------------------------------------------------------
 * (Optional) Constants & class includes
 * ---------------------------------------------------------------------- */
if ( ! defined('FL_CHILD_THEME_DIR') ) define('FL_CHILD_THEME_DIR', get_stylesheet_directory());
if ( ! defined('FL_CHILD_THEME_URL') ) define('FL_CHILD_THEME_URL', get_stylesheet_directory_uri());

// If you need the class for other utilities, you can keep the include.
// Make sure that class file does NOT enqueue styles/scripts, or comment it out.
// require_once FL_CHILD_THEME_DIR . '/classes/class-fl-child-theme.php';


/* ------------------------------------------------------------------------
 * Main Enqueue — one single block that runs very late (prints last)
 * ---------------------------------------------------------------------- */
add_action('wp_enqueue_scripts', function () {

  /* ---- 0) (Optional) Dequeue parent if theme already enqueued it under another handle ----
   * View Source to see the real handle if needed. Common guesses are listed below.
   * This prevents duplicate parent CSS printing after your child CSS.
   */
  $maybe_parent_handles = [
    'envo-royal-style',
    'envo-royal',
    'parent-style',
  ];
  foreach ( $maybe_parent_handles as $h ) {
    if ( wp_style_is( $h, 'enqueued' ) || wp_style_is( $h, 'registered' ) ) {
      wp_dequeue_style( $h );
      wp_deregister_style( $h );
    }
  }

  /* ---- 1) Enqueue Parent CSS once ---- */
  $parent_file = get_template_directory() . '/style.css';
  wp_enqueue_style(
    'parent-style',
    get_template_directory_uri() . '/style.css',
    [],
    file_exists($parent_file) ? filemtime($parent_file) : wp_get_theme( get_template() )->get('Version')
  );

  /* ---- 2) Enqueue Child CSS after Parent (updates on every save) ---- */
  $child_file = get_stylesheet_directory() . '/style.css';
  wp_enqueue_style(
    'child-style',
    get_stylesheet_uri(),
    ['parent-style'],
    file_exists($child_file) ? filemtime($child_file) : wp_get_theme()->get('Version')
  );

  /* ---- 3) Optional final overrides file (prints last; great for BB overrides) ----
   * Create this file if you want a guaranteed “wins-last” layer:
   * /wp-content/themes/envo-royal-child/assets/css/overrides.css
   */
  $overrides_file = get_stylesheet_directory() . '/assets/css/overrides.css';
  if ( file_exists($overrides_file) ) {
    wp_enqueue_style(
      'az-child-overrides',
      get_stylesheet_directory_uri() . '/assets/css/overrides.css',
      ['child-style'],
      filemtime($overrides_file)
    );
  }

  /* ---- 4) JS (put ALL child JS here; not at file top) ---- */
  $repeat_js = get_stylesheet_directory() . '/js/repeat-anim.js';
  if ( file_exists($repeat_js) ) {
    wp_enqueue_script(
      'repeat-anim',
      get_stylesheet_directory_uri() . '/js/repeat-anim.js',
      ['jquery'],                // add deps if needed
      filemtime($repeat_js),     // cache-bust on save
      true                       // in footer
    );
  }

  $custom_js = get_stylesheet_directory() . '/js/custom.js';
  if ( file_exists($custom_js) ) {
    wp_enqueue_script(
      'slide-cart',
      get_stylesheet_directory_uri() . '/js/custom.js',
      ['jquery'],
      filemtime($custom_js),
      true
    );
  }

}, 999); // VERY late so your CSS prints after theme/BB assets


/* ------------------------------------------------------------------------
 * Dev-only cache buster for styles (safe to remove in production)
 * ---------------------------------------------------------------------- */
if ( defined('WP_DEBUG') && WP_DEBUG ) {
  add_filter('style_loader_src', function($src, $handle){
    if ( in_array($handle, ['child-style','az-child-overrides'], true) ) {
      $src = add_query_arg('devv', time(), $src);
    }
    return $src;
  }, 10, 2);
}


/* ------------------------------------------------------------------------
 * Media Attachment Custom Field: "category"
 * ---------------------------------------------------------------------- */
add_filter('attachment_fields_to_edit', function($form_fields, $post){
  $value = get_post_meta($post->ID, 'category', true);
  $form_fields['category'] = [
    'label' => 'Category',
    'input' => 'text',
    'value' => $value,
    'helps' => 'Used for gallery filtering (e.g., car, house)',
  ];
  return $form_fields;
}, 10, 2);

add_filter('attachment_fields_to_save', function($post, $attachment){
  if (isset($attachment['category'])) {
    update_post_meta($post['ID'], 'category', sanitize_text_field($attachment['category']));
  }
  return $post;
}, 10, 2);


/* ------------------------------------------------------------------------
 * Shortcodes
 * ---------------------------------------------------------------------- */
add_shortcode('current_year', function () {
  return date_i18n('Y');
});
