<?php
/**
 * Frontend template — AJAX filtering (no page refresh)
 */

$settings      = $module->settings;
$module_id     = 'mod_' . $module->node;
$layout        = !empty($settings->layout_type)   ? $settings->layout_type   : 'grid';
$post_type     = !empty($settings->post_type)     ? $settings->post_type     : 'business-matchmaking';
$per_page      = !empty($settings->posts_count)   ? (int) $settings->posts_count : 6;
$custom_layout = !empty($settings->custom_layout_code) ? $settings->custom_layout_code : '';
$show_filters  = (isset($settings->show_front_filters) && 'yes' === $settings->show_front_filters);

// initial query for first render
$args = [
	'post_type'      => $post_type,
	'posts_per_page' => $per_page,
	'paged'          => 1,
	'orderby'        => 'date',
	'order'          => 'DESC',
];
$loop = new WP_Query($args);

// Nonce for AJAX
$nonce = wp_create_nonce('az_pg_nonce');

// Helper render (uses az_render_tokens from the module file)
if ( ! function_exists('az_pg_resolver_for_current_post') ) {
	function az_pg_resolver_for_current_post() {
		$acf_email        = get_field('email')         ?: '';
		$acf_budget_range = get_field('budget_range')  ?: '';
		$acf_location     = get_field('location')      ?: '';
		$acf_summary      = get_field('short_summary') ?: '';

		$ind_terms = get_the_terms(get_the_ID(), 'category-industry');
		$pt_terms  = get_the_terms(get_the_ID(), 'partnership-type');
		$ind_str   = $ind_terms ? implode(', ', wp_list_pluck($ind_terms, 'name')) : '';
		$pt_str    = $pt_terms  ? implode(', ', wp_list_pluck($pt_terms,  'name')) : '';

		$company  = '';
		$owner_id = function_exists('get_field') ? (int) get_field('bm_owner', get_the_ID()) : 0;
		if ($owner_id) {
			$company = get_user_meta($owner_id, 'company_name', true);
			if (! $company) {
				$company = get_the_author_meta('display_name', $owner_id);
			}
		}

		return [
			'post_title'         => esc_html(get_the_title()),
			'post_excerpt'       => esc_html(wp_trim_words(get_the_excerpt(), 20)),
			'post_permalink'     => esc_url(get_permalink()),
			'post_date'          => esc_html(get_the_date('M j, Y')),
			'post_thumbnail'     => get_the_post_thumbnail(null, 'large'),
			'post_thumbnail_url' => esc_url(get_the_post_thumbnail_url(null, 'large')),

			'acf_email'         => esc_html($acf_email),
			'acf_budget_range'  => esc_html($acf_budget_range),
			'acf_location'      => esc_html($acf_location),
			// Trim ACF short summary to 20 words with ellipsis
'acf_short_summary' => esc_html(
    wp_trim_words( strip_tags( $acf_summary ), 20, '…' )
),

			'acf_email_link'    => $acf_email ? '<a class="bm-mail" href="mailto:' . esc_attr($acf_email) . '">Email</a>' : '',

			'tax_industry'      => esc_html($ind_str),
			'tax_partnership'   => esc_html($pt_str),
			'company_name'      => esc_html($company),
		];
	}
}
?>
<link rel="stylesheet" href="<?php echo esc_url($module->url . 'css/az-post-grid-custom.css'); ?>" />

<section
	id="<?php echo esc_attr($module_id); ?>"
	class="az-post-grid-custom"
	data-module-id="<?php echo esc_attr($module_id); ?>"
	data-ajax-url="<?php echo esc_url(admin_url('admin-ajax.php')); ?>"
	data-nonce="<?php echo esc_attr($nonce); ?>"
	data-post-type="<?php echo esc_attr($post_type); ?>"
	data-per-page="<?php echo esc_attr($per_page); ?>"
	data-layout="<?php echo esc_attr($layout); ?>"
	data-tmpl="<?php echo esc_attr(base64_encode($custom_layout)); ?>"
>
	<?php if ($show_filters) : ?>
	<div class="az-filter-bar">
		<label class="az-filter-label" for="<?php echo esc_attr($module_id); ?>-industry">Category Industry</label>
		<select id="<?php echo esc_attr($module_id); ?>-industry" class="az-filter-select" data-param="industry">
			<option value=""><?php esc_html_e('All', 'az'); ?></option>
			<?php
			$inds = get_terms([
				'taxonomy'   => 'category-industry',
				'hide_empty' => false,
				'parent'     => 0,
			]);
			if (! empty($inds) && ! is_wp_error($inds)) {
				foreach ($inds as $t) {
					echo '<option value="' . esc_attr($t->slug) . '">' . esc_html($t->name) . '</option>';
				}
			}
			?>
		</select>

		<label class="az-filter-label" for="<?php echo esc_attr($module_id); ?>-ptype">Partnership Type</label>
		<select id="<?php echo esc_attr($module_id); ?>-ptype" class="az-filter-select" data-param="ptype">
			<option value=""><?php esc_html_e('All', 'az'); ?></option>
			<?php
			$ptypes = get_terms([
				'taxonomy'   => 'partnership-type',
				'hide_empty' => false,
				'parent'     => 0,
			]);
			if (! empty($ptypes) && ! is_wp_error($ptypes)) {
				foreach ($ptypes as $t) {
					echo '<option value="' . esc_attr($t->slug) . '">' . esc_html($t->name) . '</option>';
				}
			}
			?>
		</select>
	</div>
	<style>
		.az-filter-bar{display:flex;flex-wrap:wrap;gap:12px;align-items:center;margin:0 0 18px}
		.az-filter-label{font-weight:600;margin-right:6px}
		.az-filter-select{min-width:200px;padding:6px 8px}
	</style>
	<?php endif; ?>

	<div class="az-post-grid-wrapper grid">
		<div class="az-grid-layout" data-page="1" data-max="<?php echo esc_attr($loop->max_num_pages); ?>">
			<?php if ($loop->have_posts()) : ?>
				<?php while ($loop->have_posts()) : $loop->the_post(); ?>
					<div class="az-grid-item">
						<?php
						if ($custom_layout) {
							echo az_render_tokens($custom_layout, az_pg_resolver_for_current_post()); // phpcs:ignore WordPress.Security.EscapeOutput
						} else {
							?>
							<a class="post-thumb" href="<?php the_permalink(); ?>">
								<?php the_post_thumbnail('large'); ?>
							</a>
							<h3 class="title"><a href="<?php the_permalink(); ?>"><?php the_title(); ?></a></h3>
							<p class="excerpt"><?php echo esc_html(wp_trim_words(get_the_excerpt(), 20)); ?></p>
							<?php
						}
						?>
					</div>
				<?php endwhile; wp_reset_postdata(); ?>
			<?php else : ?>
				<div class="az-no-posts"><p><?php esc_html_e('No posts found.', 'az'); ?></p></div>
			<?php endif; ?>
		</div>

		<div class="az-load-more-wrap">
			<button class="az-load-more" <?php echo ($loop->max_num_pages <= 1) ? 'style="display:none"' : ''; ?>>
				<?php echo ! empty($settings->load_more_text) ? esc_html($settings->load_more_text) : esc_html__('Load More', 'az'); ?>
			</button>
		</div>
	</div>
</section>

<script>
(function(){
	const wrap = document.getElementById('<?php echo esc_js($module_id); ?>');
	if(!wrap) return;

	const ajaxURL  = wrap.dataset.ajaxUrl;
	const nonce    = wrap.dataset.nonce;
	const postType = wrap.dataset.postType;
	const perPage  = parseInt(wrap.dataset.perPage,10);
	const layout   = wrap.dataset.layout;
	const tmpl     = wrap.dataset.tmpl; // base64 of custom layout (may be empty)

	const grid     = wrap.querySelector('.az-grid-layout');
	const loadMore = wrap.querySelector('.az-load-more');
	const selects  = wrap.querySelectorAll('.az-filter-select');

	let filters = { industry:'', ptype:'' };

	function stateFromSelects(){
		selects.forEach(s => { filters[s.dataset.param] = s.value; });
	}

	function fetchPage(page, append=false){
		const body = new FormData();
		body.append('action', 'az_pg_fetch');
		body.append('security', nonce);
		body.append('post_type', postType);
		body.append('per_page', perPage);
		body.append('page', page);
		body.append('industry', filters.industry);
		body.append('ptype', filters.ptype);
		body.append('layout', layout);
		body.append('tmpl', tmpl);

		grid.classList.add('is-loading');

		fetch(ajaxURL, { method:'POST', body })
			.then(r => r.json())
			.then(data => {
				if(!data || !data.success){ throw new Error('Bad response'); }
				if(!append){
					grid.innerHTML = data.html;
					grid.dataset.page = '1';
				}else{
					grid.insertAdjacentHTML('beforeend', data.html);
				}
				grid.dataset.page = String(page);
				grid.dataset.max  = String(data.max_pages || 1);
				loadMore.style.display = (page >= (data.max_pages || 1)) ? 'none' : '';
			})
			.catch(()=>{})
			.finally(()=> grid.classList.remove('is-loading'));
	}

	selects.forEach(sel => sel.addEventListener('change', function(){
		stateFromSelects();
		fetchPage(1, false);
	}));

	if(loadMore){
		loadMore.addEventListener('click', function(){
			const next = parseInt(grid.dataset.page || '1', 10) + 1;
			fetchPage(next, true);
		});
	}
})();
</script>
