<?php
/**
 * Plugin Name: Custom Product Grid Widget Pro (AJAX)
 * Description: AJAX-powered product grid with badges, filters, sorting by name, and load more functionality.
 * Version: 1.6
 * Author: ChatGPT
 */

if (!defined('ABSPATH')) exit;

// Enqueue JS
function custom_product_grid_enqueue_scripts() {
    wp_enqueue_script('custom-product-grid-ajax', plugin_dir_url(__FILE__) . 'script.js', ['jquery'], null, true);
    wp_localize_script('custom-product-grid-ajax', 'customGridAjax', [
        'ajax_url' => admin_url('admin-ajax.php'),
        'nonce'    => wp_create_nonce('custom_grid_nonce')
    ]);
}
add_action('wp_enqueue_scripts', 'custom_product_grid_enqueue_scripts');

// Render shortcode
function render_custom_product_grid($atts) {
    ob_start();

    $order = isset($_GET['sort_order']) && strtolower($_GET['sort_order']) === 'desc' ? 'DESC' : 'ASC';
    $per_page = isset($_GET['per_page']) ? intval($_GET['per_page']) : 9;

    echo '<form method="get" class="sort-show-form" style="display:flex; gap:20px; align-items:center; margin-bottom:20px;">';

    echo '<label for="sort_order" style="font-weight:600;">Sort By:</label>';
    echo '<select name="sort_order" id="sort_order" onchange="this.form.submit()">';
    echo '<option value="asc"' . selected($order, 'ASC', false) . '>Ascending</option>';
    echo '<option value="desc"' . selected($order, 'DESC', false) . '>Descending</option>';
    echo '</select>';

    echo '<label for="per_page" style="font-weight:600;">Show:</label>';
    echo '<select name="per_page" id="per_page" onchange="this.form.submit()">';
    foreach ([3, 6, 9, 15, 30] as $count) {
        echo '<option value="' . $count . '"' . selected($per_page, $count, false) . '>' . $count . '</option>';
    }
    echo '</select>';

    echo '</form>';

    echo '<div id="custom-product-wrapper">';
    echo get_custom_product_grid_html(1, $order, $per_page);
    echo '</div>';
    echo '<div class="view-more"><button id="load-more-products">View More</button></div>';
    return ob_get_clean();
}
add_shortcode('custom_product_grid', 'render_custom_product_grid');

// Product grid function
function get_custom_product_grid_html($paged = 1, $order = 'ASC', $per_page = 9) {
    ob_start();

    $cat_slug = '';
    if (is_tax('product_cat')) {
        $term = get_queried_object();
        if ($term && !is_wp_error($term)) {
            $cat_slug = $term->slug;
        }
    }

    $args = [
        'post_type'      => 'product',
        'posts_per_page' => $per_page,
        'paged'          => $paged,
        'orderby'        => 'title',
        'order'          => $order,
    ];

    if (!empty($cat_slug)) {
        $args['tax_query'] = [[
            'taxonomy' => 'product_cat',
            'field'    => 'slug',
            'terms'    => $cat_slug,
        ]];
    }

    $loop = new WP_Query($args);

    if ($loop->have_posts()) {
        if ($paged === 1) {
            echo '<div class="custom-product-grid" data-page="' . esc_attr($paged) . '">';
        }
        while ($loop->have_posts()) : $loop->the_post();
            global $product;
            $tags = wp_get_post_terms($product->get_id(), 'product_tag', ['fields' => 'names']);
            ?>
            <div class="product-item">
                <div class="product-image">
                    <?php echo $product->get_image(); ?>
                    <?php if (in_array('Best Seller', $tags)) : ?>
                        <img src="<?php echo site_url('/wp-content/uploads/2025/06/best-seller.webp'); ?>" alt="Best Seller" class="badge-top-left-img" />
                    <?php endif; ?>
                    <?php if (in_array('Free Gift', $tags)) : ?>
                        <img src="<?php echo site_url('/wp-content/uploads/2025/06/free-gift.webp'); ?>" alt="Free Gift" class="badge-free-gift-img" />
                    <?php endif; ?>
                </div>
                <h3 class="product-title">
                    <?php echo get_the_title(); ?>
                    <span class="product-price">from <?php echo $product->get_price_html(); ?></span>
                </h3>
                <hr class="price-divider" />
                <a class="product-link" href="<?php the_permalink(); ?>">Find out more details ></a>
            </div>
            <?php
        endwhile;
        if ($paged === 1) {
            echo '</div>';
        }

        $is_last_page = $paged >= $loop->max_num_pages;
        echo '<div class="is-last-page" data-last="' . ($is_last_page ? 'true' : 'false') . '" style="display:none;"></div>';
        wp_reset_postdata();
    }

    return ob_get_clean();
}

// Inline styles
function custom_product_grid_styles() {
    echo '<style>
    .custom-product-grid {
        display: grid;
        grid-template-columns: repeat(3, 1fr);
        gap: 30px;
        margin-top: 20px;
    }
    .product-item {
        position: relative;
        text-align: center;
        padding: 10px;
        border: 1px solid #eee;
        border-radius: 10px;
    }
    .product-image {
        position: relative;
    }
    .badge-top-left-img {
        position: absolute;
        top: -30px;
        left: -50px;
        width: 90px;
        height: auto;
        z-index: 10;
        filter: drop-shadow(0 2px 4px rgba(0, 0, 0, 0.15));
        transform: rotate(-5deg);
        transition: all 0.3s ease;
    }
    @media (max-width: 768px) {
        .badge-top-left-img {
            top: -10px;
            left: -10px;
            width: 70px;
            transform: rotate(0deg);
        }
        .custom-product-grid {
            grid-template-columns: 1fr !important;
            overflow-x: hidden;
        }
        .product-item {
            margin: 0 auto;
            max-width: 90%;
        }
        .badge-free-gift-img {
            bottom: -8px;
            width: 80px;
        }
        .sort-show-form {
            flex-direction: column;
            align-items: flex-start;
            gap: 10px;
        }
        .sort-show-form select {
            width: 100%;
        }
    }
    .badge-free-gift-img {
        position: absolute;
        bottom: -12px;
        right: 20px;
        width: 150px;
        height: auto;
        z-index: 10;
        filter: drop-shadow(0 2px 4px rgba(0, 0, 0, 0.1));
        transform: rotate(3deg);
    }
    .product-title {
        font-size: 20px;
        font-weight: 600;
        margin-top: 10px;
        margin-bottom: 2px;
    }
    .product-price {
        display: block;
        font-size: 17px;
        font-weight: 400;
        color: #444;
    }
    .price-divider {
        border: none;
        border-top: 1px solid #ccc;
        margin: 10px 0;
    }
    .product-link {
        font-size: 15px;
        font-weight: 500;
        color: #6b8e23;
        text-decoration: none;
    }
    .product-link:hover {
        text-decoration: underline;
    }
    .view-more {
        text-align: center;
        margin-top: 20px;
    }
    #load-more-products {
        padding: 10px 20px;
        background-color: #000;
        color: #fff;
        border: none;
        border-radius: 5px;
        cursor: pointer;
    }
    .sort-show-form {
        display: flex;
        align-items: center;
        justify-content: center;
        gap: 10px;
        font-size: 14px;
        margin-bottom: 20px;
    }
    .sort-show-form label {
        margin-right: 4px;
        font-weight: 500;
    }
    .sort-show-form select {
        padding: 4px 8px;
        font-size: 14px;
        width: 120px;
        border: 1px solid #ccc;
        border-radius: 4px;
    }
    </style>';
}
add_action('wp_head', 'custom_product_grid_styles');

// AJAX logic
add_action('wp_ajax_load_more_products', 'ajax_load_more_products');
add_action('wp_ajax_nopriv_load_more_products', 'ajax_load_more_products');

function ajax_load_more_products() {
    check_ajax_referer('custom_grid_nonce', 'nonce');

    $page     = isset($_POST['page']) ? intval($_POST['page']) : 1;
    $order    = isset($_POST['order']) ? sanitize_text_field($_POST['order']) : 'ASC';
    $per_page = 100;

    echo get_custom_product_grid_html($page, $order, $per_page);
    wp_die();
}

// JavaScript
add_action('wp_footer', 'custom_product_grid_js');
function custom_product_grid_js() {
    ?>
    <script>
    jQuery(document).ready(function($) {
        var currentPage = 2;
        var order = $('#sort_order').val() || 'ASC';

        $('#load-more-products').on('click', function(e) {
            e.preventDefault();
            var $btn = $(this);
            $btn.text('Loading...');

            $.ajax({
                type: 'POST',
                url: customGridAjax.ajax_url,
                data: {
                    action: 'load_more_products',
                    nonce: customGridAjax.nonce,
                    page: currentPage,
                    order: order
                },
                success: function(response) {
                    if (response.trim() !== '') {
                        var $html = $('<div>' + response + '</div>');
                        var $items = $html.find('.product-item');
                        var isLast = $html.find('.is-last-page').data('last') === true;

                        $('.custom-product-grid').append($items);
                        currentPage++;
                        $btn.text('View More');

                        if (isLast) {
                            $('<style>#load-more-products { display: none !important; }</style>').appendTo('head');
                        }
                    } else {
                        $btn.hide();
                    }
                },
                error: function() {
                    $btn.text('Failed. Try Again');
                }
            });
        });
    });
    </script>
    <?php
}
?>
