<?php
/*
Plugin Name: Abandoned Cart Notifications (Structured)
Description: Sends push notifications for abandoned carts with server-side tracking for guests.
Version: 4.4
Author: LPS
*/

defined('ABSPATH') || exit;

define('ABNC_COOKIE_ID', 'ABNC_ID');
define('ABNC_COOKIE_LAST', 'ABNC_LAST_ADD_TIME');
define('ABNC_COOKIE_NOTIFY', 'ABNC_NOTIFIED');
define('ABNC_TIMEOUT', 60 * 60); //60 minutes
define('ABNC_COUPON_PREFIX', 'ABNC10_');
define('ABNC_NOTIFY_EXPIRE_SECONDS', 86400); // Cookie lifetime in seconds (1 day)

// Enqueue dev JS (adjust for prod mode)
add_action('init', function () {
    
    // Toggle between worker and browser mode
    define('ABNC_USE_WORKER', false); // Set to true to use service worker

    $script = ABNC_USE_WORKER ? 'push-worker-subscribe.js?v=9' : 'push-subscribe.js?v=9';
    
    wp_enqueue_script('abnc-js', plugin_dir_url(__FILE__) . $script, [], null, true);
});

// Handle ping, abandon check, reset
add_action('init', function () {
    if (isset($_GET['abnc_cart_ping'])) {
        $id = $_COOKIE[ABNC_COOKIE_ID] ?? bin2hex(random_bytes(4));
        setcookie(ABNC_COOKIE_ID, $id, time() + 86400, "/");
        exit("ABNC: Ping received, ID cookie set.");
    }

    if (isset($_GET['abandoned_cart'])) {
        abnc_handle_abandoned_cart();
        exit;
    }

    if (isset($_GET['abandoned_reset'])) {
        $cleared = [];
        foreach ([ABNC_COOKIE_ID, ABNC_COOKIE_LAST, ABNC_COOKIE_NOTIFY] as $c) {
            setcookie($c, '', time() - 3600, "/");
            $cleared[] = $c;
        }
        echo json_encode([
            'status' => 'RESET',
            'cleared' => $cleared
        ]);
        exit;
    }

    if (isset($_GET['abnc_click'])) {
        abnc_track_click();
        exit;
    }
});

// Track add-to-cart timestamp
add_action('woocommerce_add_to_cart', function () {
    if (!isset($_COOKIE[ABNC_COOKIE_ID])) {
        setcookie(ABNC_COOKIE_ID, bin2hex(random_bytes(4)), time() + 86400, "/");
    }
    setcookie(ABNC_COOKIE_LAST, time(), time() + 86400, "/");
}, 10, 0);

// Serve abandonment check
function abnc_handle_abandoned_cart() {
    $id = $_COOKIE[ABNC_COOKIE_ID] ?? null;
    $last = $_COOKIE[ABNC_COOKIE_LAST] ?? null;
    $notified = $_COOKIE[ABNC_COOKIE_NOTIFY] ?? null;

    if (!$id || !$last || $notified) {
        echo json_encode(['status' => 'SKIPPED']);
        return;
    }

    if (time() - intval($last) < ABNC_TIMEOUT) {
        $until_ts = intval($last) + ABNC_TIMEOUT;
        $until_iso = date('Y-m-d H:i:s', $until_ts);
        echo json_encode([
            'status' => 'WAIT',
            'until' => [
                'timestamp' => $until_ts,
                'iso' => $until_iso
            ]
        ]);
        return;
    }

    $key = 'abnc_coupon_' . $id;
    if (!get_option($key)) {
        $code = ABNC_COUPON_PREFIX . strtoupper(wp_generate_password(6, false));
        $cid = wp_insert_post([
            'post_title' => $code,
            'post_status' => 'publish',
            'post_type' => 'shop_coupon'
        ]);
        update_post_meta($cid, 'discount_type', 'percent');
        update_post_meta($cid, 'coupon_amount', '10');
        update_post_meta($cid, 'individual_use', 'yes');
        update_post_meta($cid, 'usage_limit', '1');
        update_post_meta($cid, 'expiry_date', date('Y-m-d H:i:s', strtotime('+1 day')));

        update_post_meta($cid, 'ABNC_ID', $id);
        update_post_meta($cid, 'IS_ABNC_COUPON', 'YES');
        update_option($key, $code);
    } else {
        $code = get_option($key);
        $cid = wc_get_coupon_id_by_code($code);
    }

    setcookie(ABNC_COOKIE_NOTIFY, 1, time() + ABNC_NOTIFY_EXPIRE_SECONDS, "/");

    echo json_encode([
        'status' => 'OK',
        'title' => '🛒 Still shopping?',
        'message' => "🎁 Use code: $code at checkout",
		'url' => site_url('/checkout') . '?abnc_click=1&abnc_id=' . urlencode($id) . '&coupon=' . urlencode($code)
    ]);
}

function abnc_track_click() {
    // Add coupon to cart if provided
    $coupon_code = $_GET['coupon'] ?? null;
    if ($coupon_code && function_exists('WC')) {
        WC()->cart->add_discount(sanitize_text_field($coupon_code));
        error_log("ABNC: Coupon '$coupon_code' added to cart programmatically");
    }
    $id = $_GET['abnc_id'] ?? null;
    if (!$id) {
        echo "Missing ID";
        return;
    }

    $coupon_code = get_option("abnc_coupon_$id");
    $cid = wc_get_coupon_id_by_code($coupon_code);
    if ($cid) {
        update_post_meta($cid, 'ABNC_IS_CLICKED', 'YES');
        update_post_meta($cid, 'ABNC_CLICKED_TIME', current_time('mysql'));
        if (is_user_logged_in()) {
            update_post_meta($cid, 'ABNC_USER_ID', get_current_user_id());
        }
        error_log("ABNC: Coupon $coupon_code clicked by $id");
    }

    wp_redirect(site_url('/checkout'));
    exit;
}

// Admin view for ABNC coupon tracking
add_action('admin_menu', function () {
    add_menu_page('ABNC Coupons', 'ABNC Coupons', 'manage_options', 'abnc_coupons', 'abnc_admin_page');
});

function abnc_admin_page() {
    echo '<div class="wrap"><h1>Abandoned Cart Coupons</h1><table class="widefat"><thead><tr><th>Coupon</th><th>Clicked</th><th>Click Time</th><th>User ID</th></tr></thead><tbody>';
    $args = ['post_type' => 'shop_coupon', 'posts_per_page' => -1, 'meta_key' => 'IS_ABNC_COUPON'];
    $coupons = get_posts($args);
    foreach ($coupons as $c) {
        $code = $c->post_title;
        $clicked = get_post_meta($c->ID, 'ABNC_IS_CLICKED', true);
        $time = get_post_meta($c->ID, 'ABNC_CLICKED_TIME', true);
        $uid = get_post_meta($c->ID, 'ABNC_USER_ID', true);
        echo "<tr><td>$code</td><td>$clicked</td><td>$time</td><td>$uid</td></tr>";
    }
    echo '</tbody></table></div>';
}
